<?php
namespace VM\FinancialStatementsBundle\Services\Restrictions;

use VM\ApiBundle\Entity\Company;
use VM\ApiBundle\Entity\CompanyEmployee;
use VM\ApiBundle\Entity\Department;
use VM\ApiBundle\Entity\MarketSegment;
use VM\ApiBundle\Entity\SummaryCompanyRound;
use VM\ApiBundle\Entity\User;
use VM\FinancialStatementsBundle\Services\Restrictions\DataStructure\DepartmentDS;
use VM\FinancialStatementsBundle\Services\Restrictions\DataStructure\ItemDS;
use VM\FinancialStatementsBundle\Services\Restrictions\DataStructure\RequiresForDepartmentDS;
use VM\FinancialStatementsBundle\Services\Restrictions\DataStructure\SegmentDS;
use VM\FinancialStatementsBundle\Services\Restrictions\Departments\DepartmentDefault;
use VM\FinancialStatementsBundle\Services\Restrictions\Departments\DepartmentElectronicsFinancial;
use VM\FinancialStatementsBundle\Services\Restrictions\Departments\DepartmentElectronicsImplement;
use VM\FinancialStatementsBundle\Services\Restrictions\Departments\DepartmentElectronicsInnovations;
use VM\FinancialStatementsBundle\Services\Restrictions\Departments\DepartmentElectronicsProduction;
use VM\FinancialStatementsBundle\Services\Restrictions\Departments\DepartmentElectronicsPromotion;
use VM\FinancialStatementsBundle\Services\Restrictions\Departments\DepartmentInterface;

class SegmentRestrictions extends BasicRestrictions
{
    /**
     * @var MarketSegment
     */
    public $MarketSegment;

    /**
     * @var Company
     */
    public $Company;

    public $productionAmount;

    /**
     * @var SegmentDS
     */
    public $SegmentDS;

    public $efficiencyTypesForProduction = array('PROD','FOR_PROD');

    public $countNeedSurfaceForOneManagement = 12;

    public $validRestrictions = true;


    public function setData(MarketSegment $MarketSegment,Company $Company)
    {
        $this->MarketSegment = $MarketSegment;
        $this->Company = $Company;
        $this->productionAmount = $Company->getProductionAmount();
        $this->SegmentDS = new SegmentDS();

        $CurrentRound = $this->getApiGetter()->getGroupActualRound($this->Company->getUserGroups());

        /** @var SummaryCompanyRound $SummaryCompanyRound */
        $SummaryCompanyRound = $this->Company->getSummaryCompanyRound()->filter(function(SummaryCompanyRound $SummaryCompanyRound) use($CurrentRound) {return $SummaryCompanyRound->getRound()->getSequence() == ($CurrentRound->getSequence()-1);})->first();
        $this->SegmentDS->zapasyCount = (empty($SummaryCompanyRound)) ? 0 : $SummaryCompanyRound->getZapasy();
    }

    public function setProductionAmount($productionAmount)
    {
        $this->productionAmount = $productionAmount;
    }

    /**
     *
     */
    public function generate()
    {
        $this->SegmentDS->setData($this->MarketSegment,$this->Company);
        $this->addDepartments();
        $this->validateSegment();
    }

    /**
     * @return SegmentDS
     */
    public function getResults()
    {
        return $this->SegmentDS;
    }

    /**
     *
     */
    public function addDepartments()
    {
        /** @var Department $Department */
        foreach($this->MarketSegment->getDepartments() as $Department)
        {
            $this->SegmentDS->addDepartment($this->createDepartmentDS($Department));
        }
    }

    /**
     * @param Department $Department
     * @return DepartmentDS
     */
    private function createDepartmentDS(Department $Department)
    {
        $DepartmentDS = new DepartmentDS();

        $DepartmentDS->setData($Department);
        $DepartmentDS->setEmployees($this->Company->getCompanyEmployees());

        if(in_array($DepartmentDS->efficiencyType,$this->efficiencyTypesForProduction))
        {
            $DepartmentDS->isRestrictedForProduction = true;
        }

        return $DepartmentDS;
    }

    private function validateSegment()
    {
        //ustawienie max produkcji na dzien podanej przez uzytkownika i wyliczenie equipment,employees,surfaceEquipment,surfaceEmployee
        $this->setProductionAmountForDepartments();

        //walidacja konkretnego wyposarzenia (relacja do Department)
        /** @var DepartmentDS $DepartmentDS */
        foreach($this->SegmentDS->departments as $DepartmentDS)
        {
            $this->validRestrictionsItemsForDepartment($DepartmentDS);
        }

        //walidacja typow wyposarzenia (MOTO,KOMP)
        $this->validRestrictionsItemsTypesForSegment();

        //walidacja liczby zatrudnionych pracowników w dzialach
        /** @var DepartmentDS $DepartmentDS */
        foreach($this->SegmentDS->departments as $DepartmentDS)
        {
            $this->validRestrictionsEmployeesForDepartment($DepartmentDS);
        }

        //walidacja powierzchni
        $this->validRestrictionsSurfaceForSegment();

        //ustawienie wydajności dla dzialow
        /** @var DepartmentDS $DepartmentDS */
        foreach($this->SegmentDS->departments as $DepartmentDS)
        {
            $this->SetEfficiencyForDepartment($DepartmentDS);
        }

        //walidacja dostępnej przestrzeni magazynowej
        $this->validRestrictionsForSegment();

//        if(!$this->SegmentDS->valid)
//        {
            $this->generateErrors();
//        }

        return $this->SegmentDS->valid;
    }

    private function setProductionAmountForDepartments()
    {
        $this->SegmentDS->maxPackages->productionAmount = $this->productionAmount;
        $this->SegmentDS->maxPackages->productionAmountPerDay = $this->productionAmount/21/12;
        $packagesForProduction = 0;

        /** @var DepartmentDS $DepartmentDS */
        foreach($this->SegmentDS->departments as $DepartmentDS)
        {
            //produkcja
            if(!empty($DepartmentDS->restrictions['efficiency']))
            {
                $productionAmountPerDay = ceil($this->SegmentDS->maxPackages->productionAmountPerDay);
                $restrictionsEfficiency = array_reverse($DepartmentDS->restrictions['efficiency'], true);
                $RequiresForDepartmentDS = new RequiresForDepartmentDS();

                foreach($restrictionsEfficiency as $id => $restrictionEfficiency)
                {
                    $restrictionEfficiencyPrev = isset($restrictionsEfficiency[$id-1]) ? $restrictionsEfficiency[$id-1] : 0;
                    while($productionAmountPerDay > $restrictionEfficiencyPrev)
                    {
                        $productionAmountPerDay -= $restrictionEfficiency;
                        $RequiresForDepartmentDS->packages++;
                        $RequiresForDepartmentDS->employees += $DepartmentDS->restrictions['personnel'][$id];
                        $RequiresForDepartmentDS->surfaceEquipment += $DepartmentDS->restrictions['surfaceEquipment'];
                        $RequiresForDepartmentDS->surfaceEmployee += $DepartmentDS->restrictions['surfaceEmployee'];
                    }
                }
                if($productionAmountPerDay > 0)
                {
                    $RequiresForDepartmentDS->packages++;
                    $RequiresForDepartmentDS->employees += $DepartmentDS->restrictions['personnel'][0];
                    $RequiresForDepartmentDS->surfaceEquipment += $DepartmentDS->restrictions['surfaceEquipment'];
                    $RequiresForDepartmentDS->surfaceEmployee += $DepartmentDS->restrictions['surfaceEmployee'];
                }
                $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol] = $RequiresForDepartmentDS;
                $packagesForProduction = $RequiresForDepartmentDS->packages;
            }
            //wymagania na wielkość produkcji
            if(!empty($DepartmentDS->restrictions['efficiencyForProduction']))
            {
                $packages = ceil($this->SegmentDS->maxPackages->productionAmountPerDay/$DepartmentDS->restrictions['efficiencyForProduction']);

                $RequiresForDepartmentDS = new RequiresForDepartmentDS();
                $RequiresForDepartmentDS->packages = $packages;
                $RequiresForDepartmentDS->employees += $DepartmentDS->restrictions['personnel'][0] * $packages;
                $RequiresForDepartmentDS->surfaceEmployee += $DepartmentDS->restrictions['surfaceEmployee']  * $packages;

                $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol] = $RequiresForDepartmentDS;
            }
            //nakłady na PiR w zależności od liczby pracowników
            if(!empty($DepartmentDS->restrictions['efficiencyPiR']))
            {
                $RequiresForDepartmentDS = new RequiresForDepartmentDS();

                $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol] = $RequiresForDepartmentDS;
            }
            //nakłady na BiR w zależności od liczby pracowników
            if(!empty($DepartmentDS->restrictions['efficiencyBiR']))
            {
                $RequiresForDepartmentDS = new RequiresForDepartmentDS();

                $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol] = $RequiresForDepartmentDS;
            }
            //wymagania na liczbę pracowników
            if(!empty($DepartmentDS->restrictions['efficiencyForEmployess']))
            {
                $packages = ceil($this->SegmentDS->getCountAllEmployees()/$DepartmentDS->restrictions['efficiencyForEmployess']);

                $RequiresForDepartmentDS = new RequiresForDepartmentDS();
                $RequiresForDepartmentDS->packages = $packages;
                $RequiresForDepartmentDS->employees += $DepartmentDS->restrictions['personnel'][0] * $packages;
                $RequiresForDepartmentDS->surfaceEmployee += $DepartmentDS->restrictions['surfaceEmployee']  * $packages;

                $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol] = $RequiresForDepartmentDS;
            }
        }

        $this->SegmentDS->maxPackages->setRequiresAll();
    }

    private function validRestrictionsItemsForDepartment(DepartmentDS $DepartmentDS)
    {
        $validRestrictionsItems = $this->validRestrictionsItems($DepartmentDS);
        $this->SegmentDS->updateMaxPackagesForDepartment($DepartmentDS->departmentSymbol,$validRestrictionsItems->maxPackages);

        foreach($validRestrictionsItems->errors as $error)
        {
            $this->SegmentDS->addDepartmentError($DepartmentDS->departmentSymbol,$error);
        }
        $this->SegmentDS->valid = (!$validRestrictionsItems->valid) ? $validRestrictionsItems->valid : $this->SegmentDS->valid;

        if($DepartmentDS->isRestrictedForProduction)
        {
            $this->SegmentDS->valid = (!$validRestrictionsItems->validRequired) ? $validRestrictionsItems->validRequired : $this->SegmentDS->valid;
            $this->SegmentDS->updateMaxPackagesForDepartment($DepartmentDS->departmentSymbol,$validRestrictionsItems->maxPackages);
        }
        if($DepartmentDS->efficiencyType == 'BIR')
        {
            $this->SegmentDS->updateMaxPackagesBiRForDepartment($DepartmentDS->departmentSymbol,$validRestrictionsItems->maxPackages);
        }
        if($DepartmentDS->efficiencyType == 'PIR')
        {
            $this->SegmentDS->updateMaxPackagesPiRForDepartment($DepartmentDS->departmentSymbol,$validRestrictionsItems->maxPackages);
        }
    }

    private function validRestrictionsItems(DepartmentDS $DepartmentDS)
    {
        $data = new \stdClass();
        $data->errors = array();
        $data->valid = true;
        $data->maxPackages = -1;
        $data->validRequired = true;

        //wymagane konkretne wyposazenie
        /** @var ItemDS $restrictionsItem */
        foreach($DepartmentDS->restrictionsItems as $restrictionsItem)
        {
            $item = $this->SegmentDS->getItemFromEquipment($restrictionsItem);
            $needPackages = $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol]->packages;

            if(is_null($item) && $needPackages != 0)
            {
                $data->errors[] = 'Brak: '.$restrictionsItem->name.'(wymagane: '.$needPackages * $restrictionsItem->quantity.')';
                $data->valid = $data->validRequired = false;
                $data->maxPackages = 0;
            }
            if(!is_null($item))
            {
                $item->used = $item->quantity;
                if($item->quantity < $needPackages * $restrictionsItem->quantity)
                {
                    $data->errors[] = 'Niewystarczająca liczba: '.$restrictionsItem->name.'(posiadasz:'.$item->quantity.',wymagane:'.$needPackages * $restrictionsItem->quantity.')';
                    $data->validRequired = false;
                }

                $maxPackages = ceil($item->quantity/$restrictionsItem->quantity);
                if($data->maxPackages < 0 || $data->maxPackages > $maxPackages)
                {
                    $data->maxPackages = $maxPackages;
                }
            }
        }

        return $data;
    }

    private function validRestrictionsItemsTypesForSegment()
    {
        if(!isset($this->SegmentDS->restrictionsItemsTypes))
        {
            return;
        }

        $validRestrictionsItemsTypes = $this->validRestrictionsItemsTypes();

        foreach($validRestrictionsItemsTypes as $symbol => $RestrictionsItemsType)
        {
            foreach($RestrictionsItemsType->errors as $error)
            {
                $this->SegmentDS->addItemsTypeError($symbol,$error);
            }
            if(!$RestrictionsItemsType->valid)
            {
                $this->SegmentDS->valid = (!$RestrictionsItemsType->valid) ? $RestrictionsItemsType->valid : $this->SegmentDS->valid;
            }

            //update mozliwosci produkcyjnych w dzialach z ogranczeniem do typow srodkow trwalych
            /** @var DepartmentDS $DepartmentDS */
            foreach($this->SegmentDS->departments as $DepartmentDS)
            {
                if(isset($DepartmentDS->restrictions['items'][$symbol]))
                {
                    if($DepartmentDS->isRestrictedForProduction)
                    {
                        $this->SegmentDS->updateMaxPackagesForDepartment($DepartmentDS->departmentSymbol,$RestrictionsItemsType->maxPackages);
                    }
                    if($DepartmentDS->efficiencyType == 'PIR')
                    {
//                        echo 'X'.$DepartmentDS->departmentSymbol.'-'.$DepartmentDS->Employees->countWithRestrictions.','.$RestrictionsItemsType->countFreeItems.'<br>';
//                        echo 'MAX='.$this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol]->maxPackagesPiR.'<br>';
                        $maxPackages = min($DepartmentDS->Employees->countWithRestrictions,$RestrictionsItemsType->countFreeItems);
                        $this->SegmentDS->updateMaxPackagesPiRForDepartment($DepartmentDS->departmentSymbol,$maxPackages);
                        $RestrictionsItemsType->countFreeItems -= $maxPackages;
//                        echo 'NEW MAX='.$this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol]->maxPackagesPiR.'<br>';
                    }
                    if($DepartmentDS->efficiencyType == 'BIR')
                    {
                        $maxPackages = min($DepartmentDS->Employees->countWithRestrictions,$RestrictionsItemsType->countFreeItems);
                        $this->SegmentDS->updateMaxPackagesBiRForDepartment($DepartmentDS->departmentSymbol,$maxPackages);
                        $RestrictionsItemsType->countFreeItems -= $maxPackages;
                    }
                }
            }
        }
    }

    private function validRestrictionsItemsTypes()
    {
        $data = array();

        foreach($this->SegmentDS->restrictionsItemsTypes as $symbol => $requiredCount)
        {
            $data[$symbol] = new \stdClass();
            $data[$symbol]->errors = array();
            $data[$symbol]->valid = true;
            $data[$symbol]->maxPackages = -1;
            $data[$symbol]->validRequired = true;
            $data[$symbol]->countFreeItems = 0;

            $countNotUsedItems = 0;
            $countNeedItems = 0;
            if(isset($this->SegmentDS->equipment[$symbol]))
            {
                /** @var ItemDS $equipmentForDepartment */
                foreach($this->SegmentDS->equipment[$symbol] as $equipmentForDepartment)
                {
                    $countNotUsedItems += ($equipmentForDepartment->quantity - $equipmentForDepartment->used);
                }
            }

            //odejmowanie komputerow dla zarządu
            if($symbol == 'KOMP')
            {
                $countNotUsedItems -= $this->getCountManagement();
            }

            /** @var DepartmentDS $DepartmentDS */
            foreach($this->SegmentDS->departments as $DepartmentDS)
            {
                if(isset($DepartmentDS->restrictions['items'][$symbol]))
                {
                    if($DepartmentDS->departmentSymbol == 'PRODUCTION')
                    {
                        $needCount = $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol]->packages;
                    }
                    else
                    {
                        $needCount = $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol]->employees;
                    }
                    $countNeedItems += $DepartmentDS->restrictions['items'][$symbol] * $needCount;
                }
            }

            if($countNotUsedItems == 0 && $countNeedItems != 0)
            {
                $data[$symbol]->errors[] = 'Nie posiadasz wyposażenia typu: '.$symbol.' (wymagane: '.$countNeedItems.')';
                $data[$symbol]->valid = $data[$symbol]->validRequired = false;
            }
            elseif($countNotUsedItems < $countNeedItems)
            {
                $data[$symbol]->errors[] = 'Niewystarczająca liczba wyposażenia typu: '.$symbol.' (posiadasz: '.$countNotUsedItems.',wymagane: '.$countNeedItems.')';
                $data[$symbol]->validRequired = false;
            }
            else
            {
                $data[$symbol]->countFreeItems = $countNotUsedItems - $countNeedItems;
            }

            $data[$symbol]->maxPackages = ceil($countNotUsedItems/$this->SegmentDS->restrictionsItemsTypes[$symbol]);
        }

        return $data;

    }

    private function validRestrictionsEmployeesForDepartment(DepartmentDS $DepartmentDS)
    {
        $valid = true;
        $countEmployeesWithRestrictions = $DepartmentDS->Employees->countWithRestrictions;

        if($DepartmentDS->isRestrictedForProduction)
        {
            $countNeedEmployees = $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol]->employees;

            if($countEmployeesWithRestrictions == 0 && $countNeedEmployees > 0)
            {
                $message = 'Brak zatrudnionych pracowników (wymagane: '.$countNeedEmployees.')';
                $this->SegmentDS->addDepartmentError($DepartmentDS->departmentSymbol,$message);
                $valid = false;
            }
            elseif($countEmployeesWithRestrictions < $countNeedEmployees)
            {
                $message = 'Niewystarczająca liczba pracowników (posiadasz: '.$countEmployeesWithRestrictions.',(wymagane: '.$countNeedEmployees.')';
                $this->SegmentDS->addDepartmentError($DepartmentDS->departmentSymbol,$message);
                $valid = false;
            }
            $this->SegmentDS->valid = (!$valid) ? $valid : $this->SegmentDS->valid;
        }
        elseif($DepartmentDS->efficiencyType == 'BIR')
        {
            $maxPackagesBiR = $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol]->maxPackagesBiR;
            $restrictionsPersonnel = isset($DepartmentDS->restrictions['personnel'][0]) ? $DepartmentDS->restrictions['personnel'][0] : 0;

            if($restrictionsPersonnel > 0 && $countEmployeesWithRestrictions < $restrictionsPersonnel*$maxPackagesBiR)
            {
                $newMaxPackagesBiR = ceil($countEmployeesWithRestrictions/$restrictionsPersonnel);
                $this->SegmentDS->updateMaxPackagesBiRForDepartment($DepartmentDS->departmentSymbol,$newMaxPackagesBiR);
            }
        }
        elseif($DepartmentDS->efficiencyType == 'PIR')
        {
            $maxPackagesPiR = $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol]->maxPackagesPiR;
            $restrictionsPersonnel = isset($DepartmentDS->restrictions['personnel'][0]) ? $DepartmentDS->restrictions['personnel'][0] : 0;

            if($restrictionsPersonnel > 0 && $countEmployeesWithRestrictions < $restrictionsPersonnel*$maxPackagesPiR)
            {
                $newMaxPackagesPiR = ceil($countEmployeesWithRestrictions/$restrictionsPersonnel);
                $this->SegmentDS->updateMaxPackagesPiRForDepartment($DepartmentDS->departmentSymbol,$newMaxPackagesPiR);
            }
        }

        $this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol] = array();
        $countEmployees = $DepartmentDS->Employees->countWithRestrictions;

        $countEquipment=0;
        if(!empty($DepartmentDS->restrictions['personnel']))
        {
            $restrictionsPersonnel = array_reverse($DepartmentDS->restrictions['personnel'], true);
            $maxPackages = $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol]->maxPackages;

            foreach($restrictionsPersonnel as $id => $restrictionPersonnel)
            {
                while($countEmployees >= $restrictionPersonnel && ($countEquipment <= $maxPackages || $maxPackages < 0))
                {
                    if(!isset($this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol][$id]))
                    {
                        $this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol][$id] = 0;
                    }
                    $this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol][$id]++;
                    $countEmployees -= $restrictionPersonnel;
                    $countEquipment++;
                }
            }

            if($maxPackages < 0 || $countEquipment < $maxPackages)
            {
                if($DepartmentDS->isRestrictedForProduction)
                {
                    $this->SegmentDS->updateMaxPackagesForDepartment($DepartmentDS->departmentSymbol,$countEquipment);
                }
                elseif($DepartmentDS->efficiencyType == 'BIR')
                {
                    $this->SegmentDS->updateMaxPackagesBiRForDepartment($DepartmentDS->departmentSymbol,$countEquipment);
                }
                elseif($DepartmentDS->efficiencyType == 'PIR')
                {
                    $this->SegmentDS->updateMaxPackagesPiRForDepartment($DepartmentDS->departmentSymbol,$countEquipment);
                }
            }
        }
    }

    private function validRestrictionsSurfaceForSegment()
    {
        //update łacznych wymagań
        $this->SegmentDS->maxPackages->setRequiresAll();

        $countNeedSurfaceOneEmployeeForOffice = 0;
        $countNeedSurfaceOneEquipmentForDepot = 0;

        /** @var DepartmentDS $DepartmentDS */
        foreach($this->SegmentDS->departments as $DepartmentDS)
        {
            $countNeedSurfaceOneEquipmentForDepot += $DepartmentDS->restrictions['surfaceEquipment'];
            $countNeedSurfaceOneEmployeeForOffice += $DepartmentDS->restrictions['surfaceEmployee'];
        }

        //łączna zakupiona powierzchnia biurowa
        $countAllSurfaceOffice = $this->getSurfaceForTypes(array('OFFICE'));
        //odjęcie przestrzeni biurowej dla zarzadu
        $countAllSurfaceOffice -= $this->getCountManagement()*$this->countNeedSurfaceForOneManagement;
        //wymagane powierzchnia biurowa
        $countNeedSurfaceOffice = $this->SegmentDS->maxPackages->requiresAll->surfaceEmployee;

        //walidacja przestrzeni biurowej
        if($countAllSurfaceOffice == 0 && $countNeedSurfaceOffice != 0)
        {
            $message = 'Brak powierzchni biurowej. (wymagane: '.$countNeedSurfaceOffice.')';
            $this->SegmentDS->addSurfaceError('OFFICE',$message);
            $this->SegmentDS->valid = false;
        }
        elseif($countAllSurfaceOffice < $countNeedSurfaceOffice)
        {
            $message = 'Niewystarczająca powierzchnia biur. (posiadasz: '.$countAllSurfaceOffice.',wymagane: '.$countNeedSurfaceOffice.')';
            $this->SegmentDS->addSurfaceError('OFFICE',$message);
            $this->SegmentDS->valid = false;
        }
        else
        {
            $freeSurface = $countAllSurfaceOffice - $countNeedSurfaceOffice;
            $this->setMaxPackagesForPIRiBIR($freeSurface);
        }

        if($countNeedSurfaceOneEmployeeForOffice > 0)
        {
            $countPackagesForOffice = ceil($countAllSurfaceOffice/$countNeedSurfaceOneEmployeeForOffice);
            $this->SegmentDS->updateMaxPackagesSurface($countPackagesForOffice);
        }

        //łączna zakupiona powierzchnia hal
        $countAllSurfaceDepot = $this->getSurfaceForTypes(array('DEPOT'));
        //wymagane powierzchnia hal
        $countNeedSurfaceDepot = $this->SegmentDS->maxPackages->requiresAll->surfaceEquipment;

        //walidacja przestrzeni hal
        if($countAllSurfaceDepot == 0 && $countNeedSurfaceDepot != 0)
        {
            $message = 'Brak powierzchni hal. (wymagane: '.$countNeedSurfaceDepot.')';
            $this->SegmentDS->addSurfaceError('DEPOT',$message);
            $this->SegmentDS->valid = false;
        }
        elseif($countAllSurfaceDepot < $countNeedSurfaceDepot)
        {
            $message = 'Niewystarczająca powierzchnia hal. (posiadasz: '.$countAllSurfaceDepot.',wymagane: '.$countNeedSurfaceDepot.')';
            $this->SegmentDS->addSurfaceError('DEPOT',$message);
            $this->SegmentDS->valid = false;
        }

        if($countNeedSurfaceOneEquipmentForDepot > 0)
        {
            $countPackagesForDepot = ceil($countAllSurfaceDepot/$countNeedSurfaceOneEquipmentForDepot);
            $this->SegmentDS->updateMaxPackagesSurface($countPackagesForDepot);
        }
    }

    private function setMaxPackagesForPIRiBIR($freeSurface)
    {
        $need = array('BIR'=>array(),'PIR'=>array());
        $CountNeed = array('BIR'=>0,'PIR'=>0);

        /** @var DepartmentDS $DepartmentDS */
        foreach($this->SegmentDS->departments as $DepartmentDS)
        {
            /** @var RequiresForDepartmentDS $RequiresForDepartment */
            $RequiresForDepartment = $this->SegmentDS->maxPackages->requiresForDepartments[$DepartmentDS->departmentSymbol];
            if(in_array($DepartmentDS->efficiencyType,array('BIR','PIR')))
            {
                $need[$DepartmentDS->efficiencyType][$DepartmentDS->departmentSymbol] = array(
                    'surface' => $DepartmentDS->restrictions['surfaceEmployee'],
                    'packages' => (($RequiresForDepartment->maxPackagesBiR > 0) ? $RequiresForDepartment->maxPackagesBiR : 0)
                );

                $CountNeed[$DepartmentDS->efficiencyType] += $need[$DepartmentDS->efficiencyType][$DepartmentDS->departmentSymbol]['surface'] * $need[$DepartmentDS->efficiencyType][$DepartmentDS->departmentSymbol]['packages'];
            }
        }
        $countAllNeed = $CountNeed['BIR']+$CountNeed['PIR'];

        if($freeSurface < $countAllNeed)
        {
            $freeSurfaceFor['BIR'] = ceil($CountNeed['BIR']*$freeSurface/$countAllNeed);
            $freeSurfaceFor['BIR'] -= $freeSurfaceFor['BIR']%12;
            $freeSurfaceFor['PIR'] = $freeSurface - $freeSurfaceFor['BIR'];

            foreach($need as $type => $departments)
            {
                foreach($departments as $symbol => $data)
                {
                    if($data['surface'] > 0 && $data['surface']*$data['packages'] > $freeSurfaceFor[$type])
                    {
                        $newPackages = floor($freeSurfaceFor[$type]/$data['surface']);
                        switch($type){
                            case 'BIR':$this->SegmentDS->updateMaxPackagesBiRForDepartment($symbol,$newPackages);break;
                            case 'PIR':$this->SegmentDS->updateMaxPackagesPiRForDepartment($symbol,$newPackages);break;
                        }
                        $freeSurfaceFor[$type] -= $data['surface']*$newPackages;
                    }
                    else
                    {
                        $freeSurfaceFor[$type] -= $data['surface']*$data['packages'];
                    }
                }
            }
        }
    }

    private function SetEfficiencyForDepartment(DepartmentDS $DepartmentDS)
    {
        $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol] = array();

        if(!empty($this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol]))
        {
            if(!empty($DepartmentDS->restrictions['efficiency']))
            {
                $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['prod'] = 0;
                foreach($this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol] as $id => $count)
                {
                    $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['prod'] += ((isset($DepartmentDS->restrictions['efficiency'][$id])) ? $DepartmentDS->restrictions['efficiency'][$id] : 0) * $count;
                }
            }
            if(!empty($DepartmentDS->restrictions['efficiencyBiR']) && $this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol] > 0)
            {
                $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['BiR'] = 0;
                $count = $this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol][0];
                $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['BiR'] = $count * $DepartmentDS->restrictions['efficiencyBiR'];
            }
            if(!empty($DepartmentDS->restrictions['efficiencyPiR']) && $this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol] > 0)
            {
                $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['PiR'] = 0;
                $count = $this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol][0];
                $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['PiR'] = $count * $DepartmentDS->restrictions['efficiencyPiR'];
            }
            if(!empty($DepartmentDS->restrictions['efficiencyForProduction']) && $this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol] > 0)
            {
                $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['prod'] = 0;
                $count = $this->SegmentDS->maxPackages->restrictionsPersonnel[$DepartmentDS->departmentSymbol][0];
                $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['prod'] = $count * $DepartmentDS->restrictions['efficiencyForProduction'];
            }
        }
        if($DepartmentDS->isRestrictedForProduction && !isset($this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['prod']))
        {
            $this->SegmentDS->maxPackages->efficiency[$DepartmentDS->departmentSymbol]['prod'] = 0;
        }
    }

    private function validRestrictionsForSegment()
    {
        if(isset($this->SegmentDS->restrictions['minSurface']) && $this->SegmentDS->restrictions['minSurface'] > 0
            && isset($this->SegmentDS->restrictions['production']) && $this->SegmentDS->restrictions['production'] > 0)
        {
            $minSurface = $this->SegmentDS->restrictions['minSurface'];
            $production = $this->SegmentDS->restrictions['production'];
            $productionAmount = $this->SegmentDS->maxPackages->productionAmount;

            $countAllSurfaceStockroom = $this->getSurfaceForTypes(array('STOCKROOM'));
            $countNeedSurfaceStockroom = (($productionAmount+$this->SegmentDS->zapasyCount)/$production)*$minSurface;

            if($countAllSurfaceStockroom == 0 && $countNeedSurfaceStockroom != 0)
            {
                $message = 'Brak powierzchni magazynowej. (wymagane: '.$countNeedSurfaceStockroom.')';
                $this->SegmentDS->addSurfaceError('STOCKROOM',$message);
                $this->SegmentDS->valid = false;
            }
            elseif($countAllSurfaceStockroom < $countNeedSurfaceStockroom)
            {
                $message = 'Niewystarczająca powierzchnia magazynowa. (posiadasz: '.$countAllSurfaceStockroom.',wymagane: '.$countNeedSurfaceStockroom.')';
                $this->SegmentDS->addSurfaceError('STOCKROOM',$message);
                $this->SegmentDS->valid = false;
            }

            $maxProduction = $this->SegmentDS->getMaxProductionForDay()*21*12;
            $needSurface = (($maxProduction+$this->SegmentDS->zapasyCount)/$production)*$minSurface;
            if($needSurface > $countAllSurfaceStockroom)
            {
                $newMaxPackagesSurface = $countAllSurfaceStockroom/$minSurface*$production/21/12;
                $this->SegmentDS->updateMaxPackagesSurface($newMaxPackagesSurface);
            }

        }
    }

    private function generateErrors()
    {
        /** @var DepartmentDS $DepartmentDS */
        foreach($this->SegmentDS->departments as $DepartmentDS)
        {
            if(isset($this->SegmentDS->departmentsErrors[$DepartmentDS->departmentSymbol]))
            {
                $this->SegmentDS->errors[] = 'Brak spełnionych wymagań w '.$DepartmentDS->name;
            }
        }
        foreach($this->SegmentDS->itemsTypeErrors as $symbol => $errors)
        {
            $this->SegmentDS->errors[] = 'Brak wymaganej liczby wyposażenia typu: '.$this->getApiGetter()->getAllegroCategoryItemTypeBySymbol($symbol)->getName();
        }
        foreach($this->SegmentDS->surfaceErrors as $symbol => $errors)
        {
            $this->SegmentDS->errors[] = 'Brak wymaganej liczby lokali typu: '.$this->getApiGetter()->getAllegroCategoryItemTypeBySymbol($symbol)->getName();
        }
    }

    private function getCountManagement()
    {
        return count($this->Company->getUserGroups()->getUsers()
            ->filter(function($User){
                return count($User->getCompanyRoles()->filter(function($CompanyRole){return in_array($CompanyRole->getSymbol(),array('M','C'));}));
            }));
    }

    private function getSurfaceForTypes($types)
    {
        $countSurface = 0;
        foreach($this->SegmentDS->equipment as $type => $equipmentForType)
        {
            if(in_array($type,$types))
            {
                /** @var ItemDS $ItemDS */
                foreach($equipmentForType as $ItemDS)
                {
                    $countSurface += ($ItemDS->surface*$ItemDS->quantity);
                }
            }
        }
        return $countSurface;
    }
}